(*********************************************
THexGridSpriteSurface -> TCustomGridSpriteSurface

If ShowGrid is true, the component sends a series
of OnDrawHex events, passing along the TPolygon
for the current grid cell that needs rendering,
along with the logical X,Y coordinates of the cell.
*********************************************)

unit HexGridSpriteSurface;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  TurboSprite, GridSpriteSurface, Grafix;

type

  TPolyDrawEvent = procedure( Sender: TObject; X, Y: integer; Poly: TPolygon ) of object;

  THexGridSpriteSurface = class( TCustomGridSpriteSurface )
  private
    poly: TPolygon;
    FOnPoly: TPolyDrawEvent;
    procedure setPoly( hexX, hexY: integer );
  protected
    procedure translateCoords( var X: integer; var Y: integer ); override;
    procedure drawCursor; override;
    procedure drawMouse; override;
    procedure drawGrid; override;
    procedure setCellsX( n: integer ); override;
    procedure setCellsY( n: integer ); override;
  public
    cellWidth: integer;
    cellEdge: integer;
    cellHeight: integer;
    cellHalfHeight: integer;
    cellMod: integer;
    colWidth: integer;
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
  published
    property OnDrawHex: TPolyDrawEvent read FOnPoly write FOnPoly;
  end;

procedure Register;

implementation

constructor THexGridSpriteSurface.Create( AOwner: TComponent );
var
  i: integer;
begin
  inherited Create( AOwner );
  poly := TPolygon.Create;
  for i := 1 to 6 do
    poly.addPoint( Point( 0,0 ) );
end;

destructor THexGridSpriteSurface.Destroy;
begin
  poly.Free;
  inherited Destroy;
end;

procedure THexGridSpriteSurface.translateCoords( var X: integer; var Y: integer );
var
  col, row: integer;
  top, left: integer;
  pct: single;
  cutoff, FromTop: integer;
begin
{ What Column are we in? }
  col := X div colWidth;
  left := col * colWidth;
{ What Row are we in? }
  row := Y div cellHeight;
  top := row * cellHeight;
  if col mod 2 = 1 then
    if (Y - top) > cellHalfHeight then
    begin
      Inc( top, cellHeight );
      Inc( row );
    end;
{ Are we within the left edge of the hex? }
  if (X - left) < cellEdge then
  begin
{ What % of the way are we from the center of the hex? }
    FromTop := Y - top;
    if FromTop > cellHalfHeight then
      pct := (FromTop - cellHalfHeight) / cellHalfHeight
    else
      pct := (cellHalfHeight - FromTop) / cellHalfHeight;
{ Determine the cutoff point at which we are still in the current hex }
    cutoff := Round( cellEdge * pct );
    if (X - left) < cutoff then
      Dec( col );
  end;
  X := col;
  if X mod 2 = 1 then
    Dec( row );
  Y := row;
end;

procedure THexGridSpriteSurface.drawCursor;
  procedure adjVertex( n, x, y: integer );
  begin
    poly.pts[n].X := poly.pts[n].X + x;
    poly.pts[n].Y := poly.pts[n].Y + y;
  end;
begin
  setPoly( FCursX, FCursY );
  DIBCanvas.polygon( 0, 0, poly );
  adjVertex( 0, 0, -1 );
  adjVertex( 1, 0, -1 );
  adjVertex( 2, 1, 0 );
  adjVertex( 3, 0, 1 );
  adjVertex( 4, 0, 1 );
  adjVertex( 5, -1, 0 );
  DIBCanvas.polygon( 0, 0, poly );
end;

procedure THexGridSpriteSurface.drawMouse;
begin
  setPoly( MouseX, MouseY );
  DIBCanvas.polygon( 0, 0, poly );
end;

procedure THexGridSpriteSurface.setCellsX( n: integer );
var
  nw: integer;
begin
  inherited setCellsX( n );
  if (LogicalWidth > 0) and (n > 0) then
  begin
    nw := LogicalWidth;
    nw := nw + ( LogicalWidth div 5 );
    cellWidth := nw div n;
    cellEdge := cellWidth div 5;
    colWidth := cellWidth - cellEdge;
  end;
end;

procedure THexGridSpriteSurface.setCellsY( n: integer );
var
  nh: integer;
begin
  inherited setCellsY( n );
  if (LogicalHeight > 0) and (n > 0) then
  begin
    nh := LogicalHeight;
    nh := nh - ( LogicalHeight div n ) div 2;
    cellHeight := nh div n;
    cellHalfHeight := cellHeight div 2;
    cellMod := cellHeight mod 2;
  end;
end;

procedure THexGridSpriteSurface.drawGrid;
var
  DrawX, DrawY: integer;
  hexX, hexY: integer;
  maxx, maxy: integer;
  colDec: integer;
  i, OrgHexX, OrgDrawX: integer;
begin
  maxy := OffsetY + Height + cellHeight;
  maxx := OffsetX + Width + colWidth;
  colDec := colWidth - cellEdge;
  if not Assigned( FOnPoly ) then
    Exit;
  DrawY := OffsetY - cellHeight;
  DrawX := OffsetX - colWidth;
  hexX := DrawX;
  hexY := DrawY;
  translateCoords( hexX, hexY );
  OrgHexX := hexX;
  OrgDrawX := DrawX;
  while DrawY <= maxy do
  begin
    hexX := OrgHexX;
    DrawX := OrgDrawX;
    while DrawX <= maxx do
    begin
      if (hexX >= 0) and (hexY >= 0) then
      begin
        setPoly( hexX, hexY );
        FOnPoly( self, hexX, hexY, poly );
      end;
      Inc( hexX );
      Inc( DrawX, colWidth );
    end;
    Inc( DrawY, cellHeight );
    Inc( hexY );
  end;
end;

procedure THexGridSpriteSurface.setPoly( hexX, hexY: integer );
begin
  poly.pts[0].X := hexX * colWidth + cellEdge - OffsetX;
  poly.pts[0].Y := hexY * cellHeight - OffsetY;
  if hexX mod 2 = 1 then
    Inc( poly.pts[0].Y, cellHalfHeight );
  poly.pts[1].X := poly.pts[0].X + ( colWidth - cellEdge );
  poly.pts[1].Y := poly.pts[0].Y;
  poly.pts[2].X := poly.pts[1].X + cellEdge;
  poly.pts[2].Y := poly.pts[1].Y + cellHalfHeight;
  poly.pts[3].X := poly.pts[2].X - cellEdge;
  poly.pts[3].Y := poly.pts[2].Y + cellHalfHeight + cellMod;
  poly.pts[4].X := poly.pts[3].X - ( colWidth - cellEdge );
  poly.pts[4].Y := poly.pts[3].Y;
  poly.pts[5].X := poly.pts[4].X - cellEdge;
  poly.pts[5].Y := poly.pts[4].Y - ( cellHalfHeight + cellMod );
end;

procedure Register;
begin
  RegisterComponents( 'TurboSprite', [THexGridSpriteSurface] );
end;

end.
